# IBM_PROLOG_BEGIN_TAG 
# This is an automatically generated prolog. 
#  
#  
#  
# Licensed Materials - Property of IBM 
#  
# (C) COPYRIGHT International Business Machines Corp. 1999,2002 
# All Rights Reserved 
#  
# US Government Users Restricted Rights - Use, duplication or 
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp. 
#  
# IBM_PROLOG_END_TAG 
package MC_cli_utils;
#"@(#)49   1.33   src/rsct/rmc/cli/pm/MC_cli_utils.pm.perl, rmccli, rsct_rori, roris01a 1/9/02 14:12:20"
######################################################################
#                                                                    #
# Package: MC_cli_utils.pm                                           #
#                                                                    #
# Description:                                                       #
#   This package contains utility/common subroutines for the PERL    #
#   Cluster Resource Monitoring and Control (RMC) CLI commands.      #
#                                                                    #
# Subroutines Available:                                             #
#                                                                    #
#   init_session - initializes a session with RMC.                   #
#                                                                    #
#   term_session - terminates a session with RMC.                    #
#                                                                    #
#   enumerate_resources - enumerates the resources of a resource     #
#     class using attribute selection. In simpler terms it builds    #
#     an array of all the resource handles for the specified         #
#     resource class that correspond to the specified selection      #
#     string. Convenient way of calling mc_enumerate_resources_bp    #
#     used by lsrsrc, rmrsrc, etc.                                   #
#                                                                    #
#   qdef_resource_class - queries the definition of a resource       #
#     class. Convenient way of calling mc_qdef_resource_class_bp     #
#     used by lsrsrc, lsrsrcdef, lsactdef, etc.                      #
#                                                                    #
#   qdef_resource_class_id - queries definition of a resource class  #
#     based on class ID, then builds a hash of class names keyed by  #
#     class id's.  Calls qdef_resource_class.                        #
#                                                                    #
#   get_p_attr_defs - function to get this resource's persistent     #
#     attribute definitions using mc_qdef_p_attributes_bp and build  #
#     a hash indexible via the attribute program name with the       #
#     attributes property and data_type definition stored as data.   #
#     Which attributes are returned is also control using the        #
#     req_attributes and req_properties paramaters.                  #
#                                                                    #
#   get_d_attr_defs - function to get this resource's dynamic        #
#     attribute definitions using mc_qdef_d_attributes_bp and build  #
#     a hash indexible via the attribute program name with the       #
#     attributes property and data_type definition stored as data.   #
#     Which attributes are returned is also control using the        #
#     req_attributes and req_properties paramaters.                  #
#                                                                    #
#   required_attr - function that returns whether the specified      #
#     attribute name is required. A required attribute has one       #
#     of the specified properties or is in the list of required      #
#     attributes (the attributes entered via the command line).      #
#                                                                    #
#   validate_rsrc_hndl - funtion to validate the specified resource  #
#     handle.  The resource handle is copied into a list of rsrc     #
#     handles (list of one) since that is what the mc_validate_rsrc  #
#     _hndl_bp function expects. A resource handle is valid if       #
#     the resource to which it is linked is still defined to the     #
#     subsystem (resource manager).                                  #
#                                                                    #
#   build_HoAttr - build Hash of Attributes (complex structure).     #
#     Give it one element of an attribute at a time.                 #
#     This is the format required by MC_cli_display_utils.pm.        #
#                                                                    #
#   get_local_node - returns a string that is the resolved node      #
#     name of this node (the node the command is running on).        #
#     No parameters.                                                 #
#                                                                    #
#   error_exit - performs required cleanup and exits with the        #
#     appropriate RMC CLI error exit code.                           #
#                                                                    #
#   printCIMsg - print common informational message to STDOUT.       #
#     Requires the program name and message mnemonic as inputs.      #
#                                                                    #
#   printCEMsg - print common error message to STDERR.               #
#     Requires the program name and message mnemonic as inputs.      #
#                                                                    #
# Examples:                                                          #
#   ($rc $session) = init_session();                                 #
#   $rc = term_session($session);                                    #
#   printCEMsg("EMsgCUcliBadOperand", $rsrc_class);                  #
#                                                                    #
#--------------------------------------------------------------------#
# Inputs:                                                            #
#   /usr/sbin/rsct/msgmaps/mccli.mcucli.map - message mapping        #
#                                                                    #
# Outputs:                                                           #
#   stdout - common informational messages that get displayed.       #
#   stderr - common error messages that get displayed.               #
#                                                                    #
# External References:                                               #
#   Commands: ctdspmsg                                               #
#                                                                    #
# Tab Settings:                                                      #
#   4 and tabs should be expanded to spaces before saving this file. #
#   in vi:  (:set ts=4  and   :%!expand -4)                          #
#                                                                    #
# Change Activity:                                                   #
#   990528 SAB 48419: Initial design & write.                        #
#   001115 GTM 67900: Add qdef_resource_class_id                     #
#   010213 SAB 70842: Changes to contact list for CT_CONTACT support.#
#   010406 SAB 71892: error_exit support new rc MC_CLI_NO_RSRC_FOUND.#
#   010406 SAB 72722: init_session support for CT_LOCAL_SCOPE.       #
#   011126 JAC 78604: add rmc management scope environment vars.     #
#   011214 JAC 77349: add get_local_node for rmcapi version 2 update #
#   020109 JAC 78200: modify required_attr for when req_attr is F's  #
######################################################################


use Exporter ();
@ISA = qw(Exporter);
@EXPORT_OK = qw(
    init_session
    term_session
    enumerate_resources
    qdef_resource_class
    qdef_resource_class_id
    get_p_attr_defs
    get_d_attr_defs
    required_attr
    validate_rsrc_hndl
    build_HoAttr
    get_local_node
    error_exit
    printCIMsg
    printCEMsg
);

use lib "/usr/sbin/rsct/pm";
use locale;

use CT::CT qw(
    :ct_contact_type_t
    CT_NONE
);

use CT::MC;                             # import defaults 
use CT::MC qw(
    :mc_session_opts_t
    :mc_qdef_opts_t

    enumerate_resources_bp
    qdef_resource_class_bp
    qdef_d_attribute_bp
    qdef_p_attribute_bp
    validate_rsrc_hndl_bp
    free_response
);

use CT::MCerr;                          # RMC return/error codes
use CT::RM;                             # RM  return/error codes

use MC_cli_rc qw(:return_codes);


#--------------------------------------------------------------------#
# Global Variables                                                   #
#--------------------------------------------------------------------#
$TRUE = 1;
$FALSE = 0;

$MSGCAT = "mccli.cat";                 # msg catalogue for this cmd
$MSGSET = "mccli";                     # common message set     

$CTDIR = "/usr/sbin/rsct";             # RSCT root directory
$CTBINDIR = "$CTDIR/bin";              # Cluster Bin directory path
$LSMSG = "$CTBINDIR/ctdspmsg";         # list / display message rtn
$ENV{'MSGMAPPATH'} = "$CTDIR/msgmaps"; # msg maps used by $LSMSG


#--------------------------------------------------------------------#
# Exported Subroutines (with @EXPORT_OK, -> on demand).              #
#--------------------------------------------------------------------#

#--------------------------------------------------------------------#
# Common message handling (error, informational) routines:           #
#--------------------------------------------------------------------#
#--------------------------------------------------------------------#
# init_session:  initialize a session with RMC.                      #
#   Returns a session class that contains an RMC session handle.     #
#                                                                    #
# Paramaters:                                                        #
#   None.                                                            #
#                                                                    #
# Returns:                                                           #
#   $rc             start_session return code.                       #
#   $session        initialized session handle.                      #
#                                                                    #
# Global Variables:                                                  #
#   $main::Trace      in  Print trace messages if turned on.         #
#   $main::Verbose    in  Print verbose messages if turned on.       #
#   CT_CONTACT        in  Environment variable, when set the RMC     #
#                         daemon to establish the session with.      #
#   CT_LOCAL_SCOPE    in  Environment variable, when set to 1, start #
#                         the RMC session with "local scope" instead #
#                         of the default "global scope".             #
#   CT_MANAGEMENT_SCOPE in Environment variable that defines the RMC #
#                         session type.                              #
#--------------------------------------------------------------------#
sub init_session
{
my $rc = 0;

if ($main::Verbose) {
    my $env_msg = "";
    if (defined $ENV{CT_CONTACT}) {  
        $env_msg = $env_msg . 
            "\tCT_CONTACT          = " . "$ENV{CT_CONTACT}\n";
    }
    if (defined $ENV{CT_LOCAL_SCOPE}) { 
        $env_msg = $env_msg . 
            "\tCT_LOCAL_SCOPE      = ". "$ENV{CT_LOCAL_SCOPE}\n";
    }
    if (defined $ENV{CT_MANAGEMENT_SCOPE}) { 
        $env_msg = $env_msg . 
            "\tCT_MANAGEMENT_SCOPE      = ". "$ENV{CT_MANAGEMENT_SCOPE}\n";
    }
    # CT_CLI_QUOTE_STRING is used in CT_cli_display_utils.pm 
    # encapsulate_string function but this is common place to 
    # display all environment variables that directly impact RMC CLI
    if (defined $ENV{CT_CLI_QUOTE_STRING}) {
        $env_msg = $env_msg . 
            "\tCT_CLI_QUOTE_STRING = ". "$ENV{CT_CLI_QUOTE_STRING}\n";
    }
    if (length $env_msg > 0) {
        printCIMsg("IMsgMCcliDisplayEnv", $env_msg);
    }
}

# Construct array of contact lists. The order of the types in the
# array control which RMC Daemon the RMC API will contact. 
# RMC Command line wants the CT_CONTACT environment variable to 
# override contacting the RMC Daemon on the local node when the 
# CT_CONTACT environment variable is set. 
# 1. CT_CONTACT_ENV_VAR if CT_CONTACT environment variable is 
#    set, contact the RMC daemon on the host or the ip address 
#    specified by the environment variable. If it is not set than
#    default to the local node. 
my $server_list = CT::CT::contact_t->new;
$server_list->set_array_count(1);
$server_list->set_contact_type(0, CT_CONTACT_ENV_VAR);

# Examine the CT_LOCAL_SCOPE environment variable to see if we
# need to change from the default global scope to local scope.
my $options = MC_SESSION_OPTS_NONE;     # Default, global scope 
if (defined $ENV{CT_LOCAL_SCOPE} && $ENV{CT_LOCAL_SCOPE} == 1) {
    $options = MC_SESSION_OPTS_LOCAL_SCOPE;
}

# After Distributed RMC, examine the CT_LOCAL_SCOPE environment 
# variable to see if we need to change from the default scope.
if (defined $ENV{CT_MANAGEMENT_SCOPE}) {
   if ($ENV{CT_MANAGEMENT_SCOPE} == 1) { $options = MC_SESSION_OPTS_LOCAL_SCOPE;}
   if ($ENV{CT_MANAGEMENT_SCOPE} == 2) { $options = MC_SESSION_OPTS_SR_SCOPE;}
   if ($ENV{CT_MANAGEMENT_SCOPE} == 3) { $options = MC_SESSION_OPTS_DM_SCOPE;}
}

my $session_handle = CT::MC::sess_hndl_t->new;

$main::Trace && print STDERR "Calling CT::MC::start_session \n";
$rc = CT::MC::start_session($server_list, $options, $session_handle);
$main::Trace && print STDERR "Return  CT::MC::start_session\n";

# Just create a dummy response and error to make error check happy
my $response = CT::MC::enumerate_rsp_t->new;
my $error = CT::MC::errnum_t->new;
$rc = error_check("mc_start_session", "none", $response, $rc, $error);

return ($rc, $session_handle);
}   #end init_session


#--------------------------------------------------------------------#
# term_session:  terminates a session with RMC.                      #
#                                                                    #
# Paramaters:                                                        #
#   $session   in   Session handle object.                           #
#                                                                    #
# Returns:                                                           #
#   $rc             end_session return code.                         #
#                                                                    #
# Global Variables:                                                  #
#   $main::Trace  in  Print trace messages if turned on.             #
#--------------------------------------------------------------------#
sub term_session {
my ($session_handle) = @_;

my $rc = 0;

$main::Trace && print STDERR "Calling CT::MC::end_session\n";

$rc = CT::MC::end_session($session_handle);

$main::Trace && print STDERR "Return  CT::MC::end_session\n";

if ($rc != 0) {
    my $rc_hex = sprintf "0x%8.8lx", $rc;
    printCEMsg("EMsgMCcliMCFunctionFailure", "mc_end_session",
        $rc, $rc_hex);
}

return($rc);
};  # end term_session


#--------------------------------------------------------------------#
# enumerate_resources - function to call the CT::MC::enumerate_      #
#   resources_bp extension and handle possible errors.               #
#                                                                    #
# Parameters:                                                        #
#   $session          input   RMC session handle.                    #
#   $rsrc_class       input   Resource class name of the resources   #
#                             for which resource handles are to be   #
#                             returned.                              #
#   $select_str       input   Selection string using only persistent #
#                             attributes. The resource handles for   #
#                             the resources that match this select   #
#                             string will be returned.               #
#   @$r_rsrc_handles  in/out  Reference to an array of resource      #
#                             handles. The resource handles that we  #
#                             want to remove.                        #
#                                                                    #
# Global References:                                                 #
#   None.                                                            #
#--------------------------------------------------------------------#
sub enumerate_resources
{
my ($session, $rsrc_class, $select_str, $r_rsrc_handles) = @_;
my $rc = 0;
 
# Need a Response 
my $response = CT::MC::enumerate_rsp_t->new;

# Need an Error 
my $error = CT::MC::errnum_t->new;

$main::Trace && print STDERR "Calling CT::MC::enumerate_resources_bp\n";

$rc = CT::MC::enumerate_resources_bp($session, $response, $error,
                                $rsrc_class, $select_str);

$main::Trace && print STDERR "Return  CT::MC::enumerate_resources_bp\n";

$rc = error_check("mc_enumerate_resources_bp", $rsrc_class, $response,
    $rc, $error);

# TODO build up the $r_resource_handles
# Need to step through each response
# Need to step through each resource handle for rsrc_handle_count in
# each response.
my $response_cnt = $response->array_count;
my ($r, $h, $rsrc_handle_cnt, $r_rsrc_handle_refs);

# TODO: should be checking the mc_error in each response?
for ($r = 0; $r < $response_cnt; $r++) {
    $rsrc_handle_cnt = $response->rsrc_handle_count($r);
    $r_rsrc_handle_refs = $response->rsrc_handles($r);
    for ($h = 0; $h < $rsrc_handle_cnt; $h++) {
        push @$r_rsrc_handles, $r_rsrc_handle_refs->[$h];
    }   
}

# Free the response - if there was one
if ($rc == 0 && $response->array_count > 0) {
    $rc = CT::MC::free_response($response);
}

return $rc;
}   # end enumerate_resources


#--------------------------------------------------------------------#
# qdef_resource_class - function to call the CT::MC::qdef_resource_  #
#   class_bp extension and handle possible errors.                   #
#                                                                    #
# Parameters:                                                        #
#   $session          input   RMC session handle.                    #
#   $class            input   name of the class we want the defn     #
#                             for if blank we want a list of all     #
#                             resource classes defined in the system.#
#   $response         in/out  in a new qdef_rsrc_class_rsp_t         #
#                             out this will point to the real RMC    #
#                             response from calling                  #
#                             mc_qdef_resource_class_bp              #
#   $req_desc         in      TRUE - descriptions requested          #
#                                                                    #
# Return                                                             #
#   $rc                       return code.                           #
#                                                                    #
# Global References:                                                 #
#   $Opt_All          input   TRUE if display def & description.     #
#--------------------------------------------------------------------#
sub qdef_resource_class
{
my ($session, $class, $response, $req_desc) = @_;
my $rc = 0;

my $error = CT::MC::errnum_t->new;

if ($req_desc) {
    $options = MC_QDEF_OPTS_NONE;
}
else {
    $options = MC_QDEF_OPTS_NODSCRP;
}

$main::Trace && print STDERR "Calling CT::MC::qdef_resource_class_bp\n";

$rc = CT::MC::qdef_resource_class_bp($session, $response, $error,
                                $options, $class);

$main::Trace && print STDERR "Return  CT::MC::qdef_resource_class_bp\n";

$rc = error_check("mc_qdef_resource_class_bp", $class, $response, 
    $rc, $error);

return $rc;
}   # end qdef_resource_class


#--------------------------------------------------------------------#
# qdef_resource_class_id - function to build a class_id to           #
#   class_name hash.  Calls qdef_resource_class.                     #
#                                                                    #
# Parameters:                                                        #
#   $session          input   RMC session handle.                    #
#                                                                    #
#   %$rHoClassNames   in/out  reference to class name hash keyed by  #
#                             class id                               #
#                                                                    #
# Return                                                             #
#   $rc                       return code.                           #
#                                                                    #
# Global References:                                                 #
#--------------------------------------------------------------------#
sub qdef_resource_class_id
{
my ($session, $rHoClassNames) = @_;

my $response = CT::MC::qdef_rsrc_class_rsp_t->new;

my $rc = qdef_resource_class($session, "", $response, 0);
($rc == 0) || return $rc;

# create hash class_id => class_name from response data
my $response_cnt = $response->array_count;
for (my $r = 0; $r < $response_cnt; $r++) {
    $$rHoClassNames{$response->class_id($r)} = $response->class_name($r);
}

if ($response->array_count > 0) {    
    CT::MC::free_response($response);
}                                    

return($rc);
}   # end qdef_resource_class_id


#--------------------------------------------------------------------#
# get_p_attr_defs - function to get this resource's persistent       #
#   attribute definitions using mc_qdef_p_attributes_bp and build    #
#   a hash indexible via the attribute program name with the         #
#   attributes property and data_type definition stored as data.     #
#   Only attributes that have the required properties (input) are    #
#   returned except required attributes (input) are always returned  #
#   if they are found regardless of their properties.                #
#   The list of required attributes may contain both persistent,     #
#   dynamic, and even invalid names. We have to get the persistent   #
#   attribute definitions so that we can determine which attributes  #
#   in the list of required attributes are persistent attributes.    #
#                                                                    #
# Parameters:                                                        #
#   $session          input   RMC session handle.                    #
#   $resource         input   name of the resource that we wish to   #
#                             create an instance of.                 #
#   $get_class        input   TRUE - get class resources defs.       #
#                             FALSE - get resource defs.             #
#   $req_properties   input   Only return the definition for attr    #
#                             that have these properties.            #
#   @$r_req_attributes input  Reference to list of required attrs    #
#                             that should be returned even if they   #
#                             do not have the required property.     #
#   @$rLoPAttr        in/out  Reference to persistent attribute      #
#                             names in order returned from RMC.      #
#   %$rHoPAttrDefs    in/out  Reference to the hash of persistent    #
#                             attribute definitions.                 #
# Return                                                             #
#   $rc                       return code.                           #
#                                                                    #
# Global References:                                                 #
#   None.                                                            #
#--------------------------------------------------------------------#
sub get_p_attr_defs
{
my ($session, $resource, $get_class, $req_properties, 
    $r_req_attributes, $rLoPAttr, $rHoPAttrDefs) = @_;
my $rc = 0;
my @attributes = ();

# Need to make sure this resource is defined and get various info
# associated with the persistent attributes.

# Need a Response for Persistent Attributes
my $response = CT::MC::qdef_pattr_rsp_t->new;

# Let the subroutine make the call to the extension, and
# handle all the assorted potential errors.
$rc = qdef_p_attribute($session, $resource, $get_class, \@attributes,
    $response);
($rc == 0) || error_exit($rc);

# Format the Query Definition Persistent Attributes into
# a hash where you can quickly lookup by attribute name, the 
# attributes definition (data type, default value, properties, id).
# Filter out the attributes that don't have the required property
# unless that attribute name is also in the list of required 
# attributes, always return required attributes if they are found. 
format_p_attr_defs($resource, $response, $req_properties, 
    $r_req_attributes, $rLoPAttr, $rHoPAttrDefs);

if ($response->array_count > 0) {
    $rc = CT::MC::free_response($response);
}

return($rc);
}   # end get_p_attr_defs


#--------------------------------------------------------------------#
# get_d_attr_defs - function to get this resource's dynamic          #
#   attribute definitions using mc_qdef_d_attributes_bp and build    #
#   a hash indexible via the attribute program name with the         #
#   attributes property and data_type definition stored as data.     #
#   Only attributes that have the required properties (input) are    #
#   returned except required attributes (input) are always returned  #
#   if they are found regardless of their properties.                #
#   The list of required attributes may contain both dynamic,        #
#   dynamic, and even invalid names. We have to get the dynamic      #
#   attribute definitions so that we can determine which attributes  #
#   in the list of required attributes are dynamic attributes.       #
#                                                                    #
# Parameters:                                                        #
#   $session          input   RMC session handle.                    #
#   $resource         input   name of the resource that we wish to   #
#                             create an instance of.                 #
#   $get_class        input   TRUE - get class resources defs.       #
#                             FALSE - get resource defs.             #
#   $req_properties   input   Only return the definition for attr    #
#                             that have these properties.            #
#   @$r_req_attributes input  Reference to list of required attrs    #
#                             that should be returned even if they   #
#                             do not have the required property.     #
#   @$rLoDAttr        in/out  Reference to dynamic attribute         #
#                             names in order returned from RMC.      #
#   %$rHoDAttrDefs    in/out  Reference to the hash of dynamic       #
#                             attribute definitions.                 #
# Return                                                             #
#   $rc                       return code.                           #
#                                                                    #
# Global References:                                                 #
#   None.                                                            #
#--------------------------------------------------------------------#
sub get_d_attr_defs
{
my ($session, $resource, $get_class, $req_properties, 
    $r_req_attributes, $rLoDAttr, $rHoDAttrDefs) = @_;
my $rc = 0;
my @attributes = ();

# Need to make sure this resource is defined and get various info
# associated with the dynamic attributes.

# Need a Response for Dynamic Attributes
my $response = CT::MC::qdef_dattr_rsp_t->new;

# Let the subroutine make the call to the extension, and
# handle all the assorted potential errors.
$rc = qdef_d_attribute($session, $resource, $get_class, \@attributes,
    $response);
($rc == 0) || error_exit($rc);

# Format the Query Definition Dynamic Attributes into
# a hash where you can quickly lookup by attribute name, the 
# attributes definition (data type, properties, id).
# Filter out the attributes that don't have the required property
# unless that attribute name is also in the list of required
# attributes, always return required attributes if they are found.
format_d_attr_defs($resource, $response, $req_properties,
    $r_req_attributes, $rLoDAttr, $rHoDAttrDefs);

if ($response->array_count > 0) {
    $rc = CT::MC::free_response($response);
}

return($rc);
}   # end get_d_attr_defs


#--------------------------------------------------------------------#
# required_attr - function to determine if the attribute name        #
#   is in the list of required attributes or if the attribute        #
#   has one of the required properties.                              #
#   78200- If $attr_properties is 0  there's never a match against   #
#        $req_properties (ANDed together). So if $req_properties is  #
#        x"FFFF" (set by user specifying -p0 to get all attributes), #
#        include the attribute.                                      #
#                                                                    #
#                                                                    #
# Parameters:                                                        #
#   $attr_name        input   The name of the attribute that we wish #
#                             to see if it is required.              #
#   $attr_properties  input   The attribute's properties.            #
#   @$r_req_attributes input  Reference to list of required attrs    #
#                             that should be returned even if they   #
#                             do not have the required property.     #
#   $req_properties   input   The attributes that have any of these  #
#                             required properties is required.       #
#                                                                    #
# Return                                                             #
#   $TRUE                     The attribute is required.             #
#   $FALSE                    The attribute is not required.         #
#                                                                    #
# Global References:                                                 #
#   None.                                                            #
#--------------------------------------------------------------------#
sub required_attr {
my ($attr_name, $attr_properties, $r_req_attributes, 
    $req_properties) = @_;

my $req_attr_name; 
my $attr_required = $FALSE;
foreach $req_attr_name (@$r_req_attributes) {
    if ($req_attr_name eq $attr_name) {
        $attr_required = $TRUE;
        last;
    }
}

# This if-statement is for 78200.
# if user specified -p0 (return attribute regardless of property)
# return the attribute   (command parse sets all F's value)
if ($req_properties == hex("0xFFFF")) {
        $attr_required = $TRUE;
}

# Filter out any attribute that does not have the required
# (requested) properties - unless that attribute is listed
# in the list of required (requested) attributes.
($attr_required || $attr_properties & $req_properties)
    ? return($TRUE)
    : return($FALSE);

}   # end required_attr


#--------------------------------------------------------------------#
# validate_rsrc_hndl - function to call the CT::MC::validate_        #
#   rsrc_hndl_bp extension and handle possible errors.               #
#   The difference is this function accepts 1 resource handle where  #
#   mc_validate_rsrc_hndl_bp expects an array of pointer to resource #
#   handles.                                                         #
#                                                                    #
# Parameters:                                                        #
#   $session          input   RMC session handle.                    #
#   $r_rsrc_handle    in      Reference to a resource handle, the    #
#                             resource handle we want to validate.   #
#   $rsrc_handle_str  in      String version of the resource handle  #
#                             for printing when error.               #
#                                                                    #
# Return                                                             #
#   $rc                       0 - resource handle is valid.          #
#                             other - resource handle is not valid.  #
#                                                                    #
# Global References:                                                 #
#   None.                                                            #
#--------------------------------------------------------------------#
sub validate_rsrc_hndl
{
my ($session, $r_rsrc_handle, $rsrc_handle_str) = @_;
my $rc = 0;
 
# Need a Response
my $response = CT::MC::rsrc_hndl_rsp_t->new;

# Need an Error
my $error = CT::MC::errnum_t->new;

my @rsrc_handles = ();
push @rsrc_handles, $r_rsrc_handle;

$main::Trace && print STDERR "Calling CT::MC::validate_rsrc_hndl_bp\n";

$rc = CT::MC::validate_rsrc_hndl_bp($session, $response, $error,
                                    \@rsrc_handles);

$main::Trace && print STDERR "Return  CT::MC::validate_rsrc_hndl_bp\n";

# Check for other errors
$rc = error_check("mc_validate_rsrc_hndl_bp", $rsrc_handle_str, 
    $response, $rc, $error);

# Free the response - if there was one
if ($rc == 0 && $response->array_count > 0) {
    $rc = CT::MC::free_response($response);
}

return $rc;
}   # end validate_rsrc_hndl


#--------------------------------------------------------------------#
# build_HoAttr - Build the hash of attributes one attribute name -   #
#   set of attribute elements at a time.                             #
#                                                                    #
# Parameters:                                                        #
#   $attr_name        in      Name of the attribute to which you     #
#                             want to add a new field (row of        #
#                             attribute elements.                    #
#   @$rLoAttrNames    in/out  Reference to an ordered list of        #
#                             attribute names. $attr_name will be    #
#                             added to this list if its not there.   #
#   %$rHoAttr         in/out  Reference to a hash of attributes.     #
#                             If $attr_name is not already in this   #
#                             hash it is added. The elements that    #
#                             make up this attribute are pushed onto #
#                             the array that make up the fields      #
#                             associated with this attribute name.   #
#   %$rHoAtElements   in      Reference to hash of an attribute's    #
#                             elements, all these elements make up   #
#                             one attribute:                         #
#                             {at_name}  => (mc_at_name)             #
#                             {at_id}    => (mc_at_id)               #
#                             {at_dtype} => (mc_at_dtype)            #
#                             {at_value} => (mc_at_value)            #
#   $rnum             in      Response number, different responses   #
#                             for same resource may have different   #
#                             # attributes, so may need to add empty #
#                             entries as placeholders.               #
#                                                                    #
# Return:                                                            #
#   $rc                       return code.                           #
#                                                                    #
# Global References:                                                 #
#   None.                                                            #
#--------------------------------------------------------------------#
sub build_HoAttr
{
my ($attr_name, $rLoAttrNames, $rHoAttr, $rHoAtElements, $rnum) = @_;

# Build the list of ordered attributes (@$rLoAttrNames) and a 
# hash of attributes (%$rHoAttr). We need both. The list to maintain
# the order from RMC and the hash for faster and more convenient 
# searching. This gets slightly more complicated since because of 
# "Variety", RMC may not always return us all the attributes we ask 
# for. So we will need to fill empty array entries in the hash for
# a particular attribute with elements of type CT_NONE. 
#
# Build the hash of attributes
# @HoAttr = {
#    <at_name> => [
#        {at_name  => $string,          at_id => $int,
#         at_dtype => $ct_data_type_t,  at_value => $scalar_reference},
#        {at_name  => $string,          at_id => $int,
#         at_dtype => $ct_data_type_t,  at_value => $scalar_reference},
#        ...
#    ],
# }
#
# For debugging to reference one element 
# $rHoAttr->{$attr_name}[$row]{at_value} 

my $rc = 0;

if (!(exists $$rHoAttr{$attr_name})) {
    # add the new attribute to the list of ordered
    # attributes
    push @$rLoAttrNames, $attr_name;

    # Add an empty array for this new attr name which will eventually
    # hold the hashes representing the values for this attribute.
    my @fields = ();
    $$rHoAttr{$attr_name} = [ @fields ];
}

# We may need to add some blank entries onto the array before this 
# one, if the #entries in array is not equal to the response we are 
# processing. This can happen when all the responses from a query
# do not return the same exact attribute names (Variety).

# Get the #elements/attributes that have this $attr_name
# scalar($$rHoPAttr{$attr_name}) does not work
my %elements = ();
my $rLoAttr = $$rHoAttr{$attr_name};
if ($#$rLoAttr + 1 < $rnum) {
    $elements{at_name}  = $attr_name;
    $elements{at_dtype} = CT_NONE; 
    for ($j = $#$rLoAttr + 1;  $j < $rnum; $j++) {
        push @{$$rHoAttr{$attr_name}}, { %elements };
        # TODO remove this next print after testing.
        # print "$main::$PROGNAME, build_HoAttr: WARNING attribute consistency problem detected, executing untested code path.\n";
    }
}
# Now really add this attribute's value to the HoAttr for this
# particular response. 
%elements = %$rHoAtElements;
push @{$$rHoAttr{$attr_name}}, { %elements };

return $rc;
}   # end build_HoAttr


#--------------------------------------------------------------------#
# get_local_node : Returns a string that is the resolved node name   #
#   of where the command is running.                                 #
#                                                                    #
# Paramaters:                                                        #
#   None.                                                            #
#                                                                    #
# Returns:                                                           #
#   local_node       Resolved name of the node the command is        #
#                    running on.                                     #
#                                                                    #
# Global Variables:                                                  #
#--------------------------------------------------------------------#
sub get_local_node
{
my $local_node = "";                    # node command runs on
my $hname = "";                         # used by gethost
my $haliases = "";                      # used by gethost
my $haddrtype = "";                     # used by gethost
my $hlength = 0;                        # used by gethost
my @haddrs = ();                        # used by gethost

# get the name of the node you're on
$local_node = `/bin/hostname`;
chomp($local_node);

# resolve the name
($hname, $haliases, $haddrtype, $hlength, @haddrs) = gethostbyname($local_node);
$local_node = $hname;

# return the resolved name
return($local_node);
}   #  end of get_local_node


#--------------------------------------------------------------------#
# error_exit - performs required cleanup and exits.                  #
# Parameters:                                                        #
#   $badrc            in      Bad return code - bad enough to exit   #
#                             processing of this command.            #
# Exit:                                                              #
#   1 MC_CLI_RMC_ERROR        Underlying RMC error.                  #
#   2 MC_CLI_ERROR            Unexpected error in the command script.#
#   3 MC_CLI_BAD_FLAG         Input flag error.                      #
#   4 MC_CLI_BAD_OPERAND      Input operand error.                   #
#   5 MC_CLI_USER_ERROR       User error.                            #
#   6 MC_CLI_NO_RSRC_FOUND    No resources found.                    #
#                                                                    #
# Global References:                                                 #
#   $main::Cleanup.    in     Hash indicating what needs to be       #
#                             cleaned up.                            #
#                             if {Session} defined - value is        #
#                             session that must terminate.           #
#                             terminate.                             #
#--------------------------------------------------------------------#
sub error_exit
{
my ($badrc) = @_;

# If a session with RMC was initialized terminate it 
# we do not care about the term_session return code since we  
# already had an error that is bad enough we are exiting
if ($main::Cleanup{Session}) {
    my $rc = term_session($main::Cleanup{Session});
}

SWITCH: {
    ($badrc == MC_CLI_RMC_ERROR)     && exit($badrc);
    ($badrc == MC_CLI_ERROR)         && exit($badrc);
    ($badrc == MC_CLI_BAD_FLAG)      && exit($badrc);
    ($badrc == MC_CLI_BAD_OPERAND)   && exit($badrc);
    ($badrc == MC_CLI_USER_ERROR)    && exit($badrc);
    ($badrc == MC_CLI_NO_RSRC_FOUND) && exit($badrc);
    # At this point all return codes should have been converted to
    # a valid RMC CLI return code.  But if one wasn't write an
    # error message.
    printCEMsg("EMsgMCcliBadRC", $badrc);
    exit(MC_CLI_ERROR);
}   # end switch

}   # end error_exit


#--------------------------------------------------------------------#
# printCIMsg : Calls $LSMSG to print out the common cluster          #
#   RMC CLI information messages with the required paramaters.       #
#   Messages printed to stdout.                                      #
#   This subroutine is like printIMsg except it is used to print     #
#   the common MC CLI messages which are in the mccli message set.   #
#                                                                    #
# Paramaters:                                                        #
#   $msg       in  Message mnemonic / message number in a sense.     #
#   $optargs   in  Extra arguments/parameters to send to $LSMSG.     #
#                                                                    #
# Returns:  None.                                                    #
#                                                                    #
# Global Variables:                                                  #
#   $main::Trace    in  Prints extra info when trace is on.          #
#   $LSMSG          in  Path & Command to display messages.          #
#   $MSGCAT         in  MC CLI Message catalogue.                    #
#   $MSGSET         in  MC CLI common message set "mccli".           #
#--------------------------------------------------------------------#
sub printCIMsg
{
my ($msg, @optargs) = @_;
my ($optarg, $optargs);

$main::Trace &&
    print STDERR "$LSMSG $MSGSET $MSGCAT $msg @optargs\n";

# Keep the args to LSMSG separate by separating with single quotes
# but must replace internal single quotes with blanks or get an error.
# Must escape internal double quotes for the system call.
foreach $optarg (@optargs) {
    $optarg =~ s/'/ /g;
    $optarg =~ s/"/\\"/g;
}
$optargs = "'" . join("' '",@optargs) . "'";

(scalar @optargs > 0) ?
    system "$LSMSG $MSGSET $MSGCAT $msg $optargs" :
    system "$LSMSG $MSGSET $MSGCAT $msg";

return;
}   # end printCIMsg


#--------------------------------------------------------------------#
# printCEMsg : Calls $LSMSG to print out the common cluster          #
#   RMC CLI error messages with the required paramaters.             #
#   Messages printed to stderr.                                      #
#   This subroutine is like printEMsg except it is used to print     #
#   the common MC CLI messages which are in the mccli message set    #
#   and it prefixes the message with the appropriate program name.   #
#                                                                    #
# Paramaters:                                                        #
#   $msg       in  Message mnemonic / message number in a sense.     #
#   $optargs   in  Extra arguments/parameters to send to $LSMSG.     #
#                                                                    #
# Returns:  None.                                                    #
#                                                                    #
# Global Variables:                                                  #
#   $main::Trace    in  Prints extra info when trace is on.          #
#   $main::PROGNAME in  Calling program/command for error message.   #
#   $LSMSG          in  Path and command to display messages.        #
#   $MSGCAT         in  CU CLI Message catalogue.                    #
#   $MSGSET         in  CU CLI common message set "cucli".           #
#--------------------------------------------------------------------#
sub printCEMsg
{
my ($msg, @optargs) = @_;
my ($optarg, $optargs);

$main::Trace &&
    print STDERR "$LSMSG $MSGSET $MSGCAT $msg $main::PROGNAME @optargs\n";

# Keep the args to LSMSG separate by separating with single quotes
# but must replace internal single quotes with blanks or get an error.
# Must escape internal double quotes for the system call.
foreach $optarg (@optargs) {
    $optarg =~ s/'/ /g;
    $optarg =~ s/"/\\"/g;
}
$optargs = "'" . join("' '",@optargs) . "'";

(scalar @optargs > 0) ?
    system "$LSMSG $MSGSET $MSGCAT $msg $main::PROGNAME $optargs 1>&2" :
    system "$LSMSG $MSGSET $MSGCAT $msg $main::PROGNAME 1>&2";

return;
}   # end printCEMsg


#--------------------------------------------------------------------#
# End Exported Subroutines (with @EXPORT_OK, -> on demand).          #
#--------------------------------------------------------------------#


#--------------------------------------------------------------------#
# error_check - checks the return code from the RMC function and     #
#   the error response return code.  If an error is detected         #
#   appropriate error messages will be displayed.                    #
#                                                                    #
# Parameters:                                                        #
#   $rmc_function     in      Name of the rmc function that was      #
#                             called and whose error code we are     #
#                             checking.                              #
#   $rmc_class        in      The rmc resource class name.           #
#   $response         in      RMC response.                          #
#   $rmc_rc           in      The rmc function return code.          #
#   $error            in      The error response.                    #
#                                                                    #
# Return:                                                            #
#   $rc                       return code.                           #
#                                                                    #
# Global References:                                                 #
#   None.                                                            #
#--------------------------------------------------------------------#
sub error_check
{
my ($rmc_function, $rmc_class, $response, $rmc_rc, $error) = @_;

my $rc = 0;
my $err_rc = $error->errnum();

if ($rmc_rc != 0) {
    if ($rmc_rc == MC_ESESSREFUSED) {
        my $rmc_contact = $ENV{CT_CONTACT};
        # TODO: In future use a unique message for this problem 
        # allowing for proper NLS translation of "local_node"
        # or display the error message via ct_cu.h error structure
        if (!defined $rmc_contact) {
            $rmc_contact = "\"local_node\"";
        }
        elsif ($rmc_contact ne "") {
            $rmc_contact = "CT_CONTACT=" . $rmc_contact;
        }
        else {
            $rmc_contact = "CT_CONTACT=\"\"";
        }
        printCEMsg("EMsgMCcliInvalidSess", $rmc_contact);
        $rc = MC_CLI_USER_ERROR;
    } 
    elsif ($rmc_rc == MC_EAUTHENTICATE) {
        printCEMsg("EMsgMCcliNotAuthenticated");
        $rc = MC_CLI_USER_ERROR;
    }
    elsif ($rmc_rc == MC_EAUTHORIZATION) {
        printCEMsg("EMsgMCcliNotAuthorized");
        $rc = MC_CLI_USER_ERROR;
    }
    else {
        if ($rmc_function eq "mc_enumerate_resources_bp") {
            printCEMsg("EMsgMCcliEnumRsrcError", $rmc_class);
        }
        my $rmc_rc_hex = sprintf "0x%8.8lx", $rmc_rc;
        printCEMsg("EMsgMCcliMCFunctionFailure", $rmc_function, 
            $rmc_rc, $rmc_rc_hex);
        $rc = MC_CLI_RMC_ERROR;
        return $rc;
    }
}

# Check the errnum in each of the RMC responses
for (my $r = 0; $r < $response->array_count; $r++) {
    if ($r > 0) {
        $response->error($error, $r);
        $err_rc = $error->errnum();
    }
    if ($err_rc != 0) {
        if ($err_rc == CT::MCerr::RMC_ECLASSNOTDEFINED) {
            printCEMsg("EMsgMCcliClassNotDef", $rmc_class);
            $rc = MC_CLI_USER_ERROR;
        }
        elsif ($err_rc == CT::MCerr::RMC_EBADRSRCHANDLE || 
               $err_rc == CT::RM::RM_EINVRESHANDLE ||
               $err_rc == CT::RM::RM_EINVALIDHANDLE) {
            printCEMsg("EMsgMCcliInvalidRsrcHandle", $rmc_class);
            $rc = MC_CLI_USER_ERROR;
        }
        elsif ($err_rc == CT::MCerr::RMC_EACCESS) {
            print STDERR $error->error_msg;
            $rc = MC_CLI_USER_ERROR;
        }
        elsif ($err_rc >= 0x60000 && $err_rc <= 0x6ffff) {
            # Selection string errors are in this range
            printCEMsg("EMsgMCcliSelectStrError");
            print STDERR $error->error_msg;
            $rc = MC_CLI_USER_ERROR;
        }
        else {
            if ($rmc_function eq "mc_enumerate_resources_bp") {
                printCEMsg("EMsgMCcliEnumRsrcError", $rmc_class);
            }
            my $err_rc_hex = sprintf "0x%8.8lx", $err_rc;
            printCEMsg("EMsgMCcliMCFunctionFailure", $rmc_function, 
                $err_rc, $err_rc_hex);
            print STDERR $error->error_msg;
            $rc = MC_CLI_RMC_ERROR;
        }
    }   # end if
}   # end for

return $rc;
}   # end error_check


#--------------------------------------------------------------------#
# format_p_attr_defs - formats the persistent attributes definitions #
#   into a complex hash (%rHoPAttrDefs). This hash can be used to    #
#   quickly lookup whether a particular attribute name is a          #
#   persistent attribute. In addition if more specific definition    #
#   information is needed for this attribute such as its data type,  #
#   id, properties or default value - they are all stored in the     #
#   hash.                                                            #
#   Use the following syntax to access the persistent attribute      #
#   definitions for a resource.                                      #
#   for each $attribute @$rLoPAttrDefs {                             #
#     $HoPAttrDefs{$attribute}{at_id} - attribute id.                #
#     $HoPAttrDefs{$attribute}{at_dtype} - attribute data type.      #
#     $HoPAttrDefs{$attribute}{at_properties} - attr properties.     #
#     $HoPAttrDefs{$attribute}{at_dvalue} - attr default value.      #
#   }                                                                #
#                                                                    #
# Parameters:                                                        #
#   $resource         input   Resource Name                          #
#   $response         input   Response data structure.               #
#   $req_properties   input   Requested properties - only put the    #
#                             attributes in the hash that have these #
#                             requested properties.                  #
#   @$r_req_attributes input  Reference to list of required attrs    #
#                             that should be returned even if they   #
#                             do not have the required property.     #
#   $rLoPAttr         in/out  The list of ordered persistent attr    #
#                             names in order returned in responses.  #
#   $rHoPAttrDefs     in/out  The hash of persistent attribute defs. #
#                                                                    #
# Return:                                                            #
#   None.                                                            #
#                                                                    #
# Global References:                                                 #
#   None.                                                            #
#--------------------------------------------------------------------#
sub format_p_attr_defs
{
my($resource, $response, $req_properties, $r_req_attributes, 
    $rLoPAttr, $rHoPAttrDefs) = @_;

my($attr_name, $r);

# format each of the attributes in the response (after filtering)
# into a complex hash of attribute names and definitions. Allowing
# easier access to the data... 

my %elements = ();
my $response_cnt = $response->array_count;
for ($r = 0; $r < $response_cnt; $r++) {
    %elements = ();
    my $attr_name = $response->program_name($r); 
    # Filter out any attribute that does not have the required 
    # (requested) properties - unless that attribute is listed
    # in the list of required (requested) attributes.
    (required_attr($attr_name, $response->properties($r),
        $r_req_attributes, $req_properties)) || next;

    $elements{at_name} = $attr_name; 
    $elements{at_id} = $response->attribute_id($r);
    $elements{at_properties} = $response->properties($r);
    $elements{at_dtype} = $response->data_type($r);
    $elements{at_dvalue} = $response->default_value($r);
    push @$rLoPAttr, $attr_name;
    $$rHoPAttrDefs{$attr_name} = { %elements };
}   # end for responses

return;
}   # end of format_p_attr_defs


#--------------------------------------------------------------------#
# format_d_attr_defs - formats the dynamic attributes definitions    #
#   into a complex hash (%rHoDAttrDefs). This hash can be used to    #
#   quickly lookup whether a particular attribute name is a          #
#   dynamic attribute. In addition if more specific definition       #
#   information is needed for this attribute such as its data type,  #
#   id, properties - they are all stored in the hash.                #
#   Use the following syntax to access the dynamic attribute         #
#   definitions for a resource.                                      #
#   for each $attribute @$rLoDAttrDefs {                             #
#     $HoDAttrDefs{$attribute}{at_id} - attribute id.                #
#     $HoDAttrDefs{$attribute}{at_dtype} - attribute data type.      #
#     $HoDAttrDefs{$attribute}{at_properties} - attr properties.     #
#   }                                                                #
#                                                                    #
# Parameters:                                                        #
#   $resource         input   Resource Name                          #
#   $response         input   Response data structure.               #
#   $req_properties   input   Requested properties - only put the    #
#                             attributes in the hash that have these #
#                             requested properties.                  #
#   @$r_req_attributes input  Reference to list of required attrs    #
#                             that should be returned even if they   #
#                             do not have the required property.     #
#   $rLoDAttr         in/out  The list of ordered dynamic attr       #
#                             names in order returned in responses.  #
#   $rHoDAttrDefs     in/out  The hash of dynamic attribute defs.    #
#                                                                    #
# Return:                                                            #
#   None.                                                            #
#                                                                    #
# Global References:                                                 #
#   None.                                                            #
#--------------------------------------------------------------------#
sub format_d_attr_defs
{
my($resource, $response, $req_properties, $r_req_attributes,
    $rLoDAttr, $rHoDAttrDefs) = @_;

my($attr_name, $r);

# format each of the attributes in the response (after filtering)
# into a complex hash of attribute names and definitions. Allowing
# easier access to the data... 

my %elements = ();
my $response_cnt = $response->array_count;
for ($r = 0; $r < $response_cnt; $r++) {
    %elements = ();
    my $attr_name = $response->program_name($r);
    # Filter out any attribute that does not have the required 
    # (requested) properties - unless that attribute is listed
    # in the list of required (requested) attributes.
    (required_attr($attr_name, $response->properties($r),
        $r_req_attributes, $req_properties)) || next;

    # Filter out any attribute that is a Quantum 
    # (has a data_type of CT_NONE) since we can't qurey there 
    # attr values - RMC complains.
    ($response->data_type($r) == CT_NONE) && next;

    $elements{at_name} = $attr_name; 
    $elements{at_id} = $response->attribute_id($r);
    $elements{at_properties} = $response->properties($r);
    $elements{at_dtype} = $response->data_type($r);
    push @$rLoDAttr, $attr_name;
    $$rHoDAttrDefs{$attr_name} = { %elements };
}   # end for responses

return;
}   # end of format_d_attr_defs


#--------------------------------------------------------------------#
# qdef_p_attribute - function to call the CT::MC::qdef_p_attribute_  #
#   bp extension and handle possible errors.                         #
#                                                                    #
# Parameters:                                                        #
#   $session          input   RMC session handle.                    #
#   $class            input   Name of the class we want the defn     #
#                             for.                                   #
#   $get_class        input   TRUE - get the resource class defs.    #
#                             FALSE - get the resource defs.         #
#   @$r_attributes    input   Reference to array of attribute names. #
#   $response         in/out  Response data structure, on output     #
#                             this will point to the real resposne.  #
#                                                                    #
# Return:                                                            #
#   $rc                       return code.                           #
#                                                                    #
# Global References:                                                 #
#   Trace             input  TRUE if trace calls to extensions.      #
#--------------------------------------------------------------------#
sub qdef_p_attribute
{
my ($session, $class, $get_class, $r_attributes, $response) = @_;
my $rc = 0;

my $error = CT::MC::errnum_t->new;

my $options = MC_QDEF_OPTS_NODSCRP;

my $attr_count = scalar(@$r_attributes);

$main::Trace && print STDERR "Calling CT::MC::qdef_p_attribute_bp\n";

$rc = CT::MC::qdef_p_attribute_bp($session, $response, $error,
                                $options, $class, $get_class,
                                $r_attributes, $attr_count);

$main::Trace && print STDERR "Return  CT::MC::qdef_p_attribute_bp\n";

$rc = error_check("mc_qdef_p_attribute_bp", $class, $response,
    $rc, $error);

return $rc;
}   # end qdef_p_attribute


#--------------------------------------------------------------------#
# qdef_d_attribute - function to call the CT::MC::qdef_d_attribute_  #
#   bp extension and handle possible errors.                         #
#                                                                    #
# Parameters:                                                        #
#   $session          input   RMC session handle.                    #
#   $class            input   Name of the class we want the defn     #
#                             for.                                   #
#   $get_class        input   TRUE - get the resource class defs.    #
#                             FALSE - get the resource defs.         #
#   @$r_attributes    input   Reference to array of attribute names. #
#   $response         in/out  Response data structure, on output     #
#                             this will point to the real resposne.  #
#                                                                    #
# Return:                                                            #
#   $rc                       return code.                           #
#                                                                    #
# Global References:                                                 #
#   Trace             input  TRUE if trace calls to extensions.      #
#--------------------------------------------------------------------#
sub qdef_d_attribute
{
my ($session, $class, $get_class, $r_attributes, $response) = @_;
my $rc = 0;

my $error = CT::MC::errnum_t->new;

my $options = MC_QDEF_OPTS_NODSCRP;

my $attr_count = scalar(@$r_attributes);

$main::Trace && print STDERR "Calling CT::MC::qdef_d_attribute_bp\n";

$rc = CT::MC::qdef_d_attribute_bp($session, $response, $error,
                                $options, $class, $get_class,
                                $r_attributes, $attr_count);

$main::Trace && print STDERR "Return  CT::MC::qdef_d_attribute_bp\n";

$rc = error_check("mc_qdef_d_attribute_bp", $class, $response,
    $rc, $error);

return $rc;
}   # end qdef_d_attribute


#--------------------------------------------------------------------#
# End Non Exported Subroutines - only used within this pm.           #
#--------------------------------------------------------------------#

#--------------------------------------------------------------------#
# End File.                                                          #
#--------------------------------------------------------------------#
